<?php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include_once "../db_connection.php";

// Function to resend PAN using Brevo API and log resends
function resendPAN($conn, $PAN_ID, $FormType, $secretkey, $opts = []) {
    date_default_timezone_set('America/Chicago');

    $opts = array_merge([
        'config_path' => __DIR__ . '/../config.php',
        'log_file'    => __DIR__ . '/resend_log.txt',
        'brevo_config_path' => __DIR__ . '/../vendor/PAN_EmailConfig.php',
    ], $opts);

    $PAN_ID = intval($PAN_ID);
    if ($PAN_ID <= 0) {
        return ['success' => false, 'error' => 'Invalid PAN_ID'];
    }

    // Load PAN record
    $stmt = $conn->prepare("SELECT * FROM PAN_newhires WHERE PAN_ID = ? LIMIT 1");
    if (!$stmt) return ['success'=>false,'error'=>'Prepare failed: '.$conn->error];
    $stmt->bind_param('i', $PAN_ID);
    $stmt->execute();
    $data = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    if (!$data) return ['success'=>false,'error'=>'No record found for PAN_ID '.$PAN_ID];

    // Load email config
    $PAN_EmailConfig = file_exists($opts['config_path']) ? include $opts['config_path'] : [];
    $branch = strtoupper(trim($data['branch'] ?? ''));
    $department = strtoupper(trim($data['department'] ?? ''));
    $normalizedConfig = [];

    foreach ($PAN_EmailConfig as $b => $deps) {
        if ($b === 'default') {
            $normalizedConfig['default'] = $deps;
            continue;
        }
        if (!is_array($deps)) continue;
        $bKey = strtoupper(trim($b));
        foreach ($deps as $d => $email) {
            $normalizedConfig[$bKey][strtoupper(trim($d))] = $email;
        }
    }

    // Resolve recipient email
    $to = !empty($normalizedConfig[$branch][$department])
        ? $normalizedConfig[$branch][$department]
        : ($normalizedConfig['default'] ?? null);
    if (!$to) return ['success'=>false,'error'=>'No recipient email configured for branch/department'];

    // Subject
    $subject = "PERSONNEL ACTION NOTICE - " . strtoupper(($data['first_name'] ?? '') . ' ' . ($data['last_name'] ?? ''));

    // Recipient name
    $recipientName = 'Approver';
    if (!empty($to)) {
        $emailUser = explode('@', $to)[0];
        $emailUserParts = preg_split('/[._]/', $emailUser);
        $emailUserParts = array_map('ucfirst', $emailUserParts);
        $recipientName = implode(' ', $emailUserParts);
    }

    // Build HTML email (same as before)
    $c = 0;
    $email_message = '<html><body>';
    $email_message .= '<p>Hello ' . htmlspecialchars($recipientName) . ',</p>';
    $email_message .= '<p>It has been over 24 hours since below PAN was submitted for your approval. Please Approve or Reject at your earliest convenience.</p>';

    $email_message .= '<table rules="all" style="border-color: #666; font-family: Montserrat-Regular; font-size:14px; border-radius:7px 7px 0 0" cellpadding="10">';
    $email_message .= '<tr style="background: #fff"><td colspan="2" align="center"><img src="http://www.selectransportation.com/helpdesk/Images/STR-LOGO1.png"></td></tr>';
    $email_message .= '<tr style="background: #000000; color:#fff; border-radius: 8px 8px 0 0"><td colspan="2" align="center"><h3>' . strip_tags($subject) .'</h3></td></tr>';

    $personnelNoticeList = [
        'PAN_ID'           => $data['PAN_ID'],
        'first_name'       => $data['first_name'],
        'last_name'        => $data['last_name'],
        'effective_date'   => $data['effective_date'],
        'branch'           => $data['branch'],
        'department'       => $data['department'],
        'employeeID'       => $data['employeeID'],
        'address'          => $data['address'],
        'city'             => $data['city'],
        'state'            => $data['state'],
        'zip'              => $data['zip'],
        'telephone'        => $data['telephone']
    ];

    foreach ($personnelNoticeList as $key => $val) {
        if (isset($val) && $val != '') {
            $colour = ($c % 2 == 0) ? '#ffffff' : '#f3f3f3';
            $email_message .= '<tr bgcolor="' . $colour . '">';
            $email_message .= '<td><strong>'. ucwords(str_replace("_", " ", $key))  . '</strong></td>';
            $email_message .= '<td>'. wordwrap(htmlspecialchars($val, ENT_QUOTES),75, "<br />\n") .'</td>';
            $email_message .= '</tr>';
            $c++;
        }
    }

    // Approve/Reject button
    $email_message .= '<tr bgcolor="#ccc">';
    $email_message .= '<td colspan="2" width="100%" height="30" bgcolor="#1F7F4C" style="text-align:center; color: #ffffff;">';
    $email_message .= '<a href="https://helpdesk.selectransportation.com/approval.php?PAN_ID=' . base64_encode($PAN_ID) .  '&personnelOnly=' . base64_encode($FormType) .'" style="font-size:15px; font-weight: bold; font-family: Helvetica, Arial, sans-serif; text-decoration: none; line-height:40px; width:100%; display:inline-block"><span style="color: #FFFFFF">APPROVE/REJECT</span></a>';
    $email_message .= '</td></tr>';

    $email_message .= "<tr style='background: #000000; color:#fff; border-radius: 8px 8px 0 0'><td colspan='2' align='center'>Copyright &copy STR HelpDesk. All Rights Reserved.</td></tr>";
    $email_message .= "</table></body></html>";

    // From email/name
    $fromEmail = !empty($data['from_email']) ? $data['from_email'] : 'no-reply@selectransportation.com';
    $fromName  = !empty($data['fromName']) ? $data['fromName'] : 'STR HelpDesk';

    // 🔑 Load Brevo API key (support both define() and array return)
    $brevoApiKey = null;
    if (file_exists($opts['brevo_config_path'])) {
        $config = include $opts['brevo_config_path'];
        if (is_array($config) && !empty($config['BREVO_API_KEY'])) {
            $brevoApiKey = $config['BREVO_API_KEY'];
        } elseif (defined('BREVO_API_KEY')) {
            $brevoApiKey = BREVO_API_KEY;
        }
    } elseif (defined('BREVO_API_KEY')) {
        $brevoApiKey = BREVO_API_KEY;
    }

    if (!$brevoApiKey) {
        return ['success'=>false, 'error'=>'Brevo API key not found'];
    }

    // Build payload for Brevo API
    $payload = [
        'sender' => ['name' => $fromName, 'email' => $fromEmail],
        'to'     => [['email' => $to, 'name' => $recipientName]],
        'subject'=> $subject,
        'htmlContent' => $email_message
    ];

    // Call Brevo
    $ch = curl_init('https://api.brevo.com/v3/smtp/email');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'accept: application/json',
        'content-type: application/json',
        'api-key: '.$brevoApiKey
    ]);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

    $response = curl_exec($ch);
    $curlErr = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $success = ($httpCode >= 200 && $httpCode < 300);

    // 🔎 Log full API response for debugging
    $logLine = date('Y-m-d H:i:s') . " | PAN_ID: $PAN_ID | To: $to | Success: " . ($success ? 'Yes' : "No") . " | HTTP $httpCode | Response: $response | CurlErr: $curlErr\n";
    file_put_contents($opts['log_file'], $logLine, FILE_APPEND);

    return $success
        ? ['success'=>true, 'to'=>$to, 'subject'=>$subject]
        : ['success'=>false, 'error'=>"Brevo send failed (HTTP $httpCode) Response: $response Curl: $curlErr", 'to'=>$to, 'subject'=>$subject];
}
